---
title: "Upload Asset File"
order: 1
---

*Upload Asset File* is available as both a `pgutil` command and an HTTP Request, and will create a new file in the asset directory, or overwrite an existing one.

:::(Info) (🚀 Quick Example: Uploading a file with pgutil)
This example will upload a file `download-stats.txt` located in the local path `C:\Inedo\data-files` the folder `uploaded-files` of the asset directory `myAssetDirectory`:

```bash
pgutil assets upload --feed=myAssetDirectory --target-path=uploaded-files/download-stats.txt --file=C:\Inedo\data-files\download-stats.txt
```
:::

## Command Specification (CLI)
The `assets upload` command is used to upload a file to the asset directory.

The `--target-path` and `--file` options are always required. The `--feed` option is required if there is no default feed configured. 

**Uploading a file** requires the asset directory (e.g. `MyAssetDirectory`), the target folder in the asset directory to upload to (e.g. `uploaded-files`), and the path of the file to upload (e.g. `C:\Inedo\data-files\download-stats.txt`):

```bash
pgutil assets upload --feed=myAssetDirectory --target-path=uploaded-files/download-stats.txt --file=C:\Inedo\data-files\download-stats.txt
```

The `--partsize` option is required when multipart uploading, and states the size of each part. 

**Uploading a mutipart file** requires the asset directory (e.g. `MyAssetDirectory`), the target folder in the asset directory to upload to (e.g. `uploaded-files`), the path of the file to upload (e.g. `C:\Inedo\data-files\download-stats.txt`) and the size in mb of the part size of each upload (e.g. `2048` for a 2gb file):

```bash
pgutil assets upload --feed=myAssetDirectory --target-path=uploaded-files/download-stats.txt --file=C:\Inedo\data-files\download-stats.txt --partsize=2048
```

## HTTP Request Specification
This HTTP request uses `PUT`, `POST` and `PATCH` requests. The behaviour is determined by the request used:

* `PUT` will only upload a file if it doesn't already exist.
* `POST` will create a file in the destination if it doesn't currently exist, or overwrite any existing file with the same name.
* `PATCH` will only upload the file if a file with the same name currently exists, overwriting it.

To create or overwrite a file, simply `PUT`, `POST` or `PATCH` to the URL with the `AssetDirectoryName`, the `file path`, and an [appropriate API key](/docs/proget/api/assets#authentication).

```bash
PUT/POST/PATCH /endpoints/«AssetDirectoryName»/content/«path_to_file»
```

If the request specifies a folder that does not currently exist, the folder will be created.

#### Multi-part Upload: Large Files
There is an HTTP request that can be used for very large (2GB+) files. Please see the [Multipart Asset File Upload](/docs/proget/api/assets/files/upload/multipart) page for this.

## HTTP Response Specification
A successful `201` response will save the file to the specified asset directory. A `401` response indicates a [missing, unknown, or unauthorized API Key](/docs/proget/api/assets#authentication).

## Sample Usage Scripts

### Upload all files in a folder (Powershell)
This script uploads all files within a folder, `C:\Proget\upload-files` to a folder, `uploaded`, in an asset directory, `internal-files`:

:::(info)
This can be used when bulk uploading individual files located in a folder. For a more efficient solution of uploading a batch of files within an archive, use our [Import Archive](/docs/proget/api/assets/folders/import) endpoint.
:::

```powershell
$apiKey = "a1b2c3d4e5"
$baseUrl = "https://proget.corp.local"
$assetdirectory = "internal-files"
$folder = "uploaded"
$localFolder = "C:\Proget\upload-files"

$headers = @{
    "Content-Type" = "application/octet-stream"
    "X-ApiKey" = $apiKey
}

Get-ChildItem -Path $localFolder | ForEach-Object {
    $fileName = $_.Name
    $apiEndpoint = "$baseUrl/$assetdirectory/content/$folder/$fileName"
    $fileContent = [System.IO.File]::ReadAllBytes($_.FullName)

    Invoke-RestMethod -Uri $apiEndpoint -Method Put -Headers $headers -Body $fileContent -ContentType "application/octet-stream"

    Write-Host "File '$fileName' uploaded to $apiEndpoint"
}
```

#### Example Output:

```powershell
File 'assetcontent.bin' uploaded to https://proget.corp.local/endpoints/internal-files/content/uploaded/assetcontent.bin
File 'readme.txt' uploaded to https://proget.corp.local/endpoints/internal-files/content/uploaded/readme.txt
File 'logo.png' uploaded to https://proget.corp.local/endpoints/internal-files/content/uploaded/logo.png
...
...
```