---
title: "Statements"
order: 1
---

In OtterScript, Statements are the core building elements that define the logic and flow of your deployment plans. The execution engine processes plans in a top-down manner, interpreting one statement at a time and managing blocks according to their structure and type.

If you're familiar with programming or scripting concepts, OtterScript’s approach will feel intuitive. However, it also introduces unique features such as log scoping, asynchronous execution, and isolated processes to better suit deployment and infrastructure automation needs.

Statements are individual actions or directives within a plan. Most statements are operations, but OtterScript also provides control and flow statements like:
* Set Variable – define or update values
* Log – write messages to the execution log
* Set Status – manually change the execution status
* Raise Error – explicitly trigger an error
* Await – pause execution to wait for asynchronous processes

Statements execute sequentially unless the plan’s flow is altered by blocks or specific flow control. There are several types of code statement:

| Statement Type | Purpose |
|------------|---------|
| [General Statements](/docs/executionengine/coreconcepts/statements/general) | Group statements with shared execution settings, like server targeting or async execution |
| [If/Else Statements](/docs/executionengine/coreconcepts/statements/if-else) | Branch execution based on a condition |
| [Foreach Statements](/docs/executionengine/coreconcepts/statements/foreach) | Loop through items like arrays or ranges |
| [Try/Catch Statements](/docs/executionengine/coreconcepts/statements/try-catch) | Handle errors gracefully and execute fallback actions |

### Statement Blocks

Statement Blocks group statements together and manage their execution context. Blocks provide:
* Flow control (e.g., conditional branching, loops, error handling)
* Variable and log scoping
* Parallel and exclusive execution
* Error isolation and retry logic

Each block introduces a new scope for both variables and logging, meaning:
* Variables declared inside a block are isolated to that block (and its nested blocks).
* Logs for actions inside a block can be collapsed or expanded in the execution logs, making plans easier to debug and understand.

## Visual and Text Modes

You can define statements either in:

* **Text Mode:** Gives full flexibility and is better suited for advanced users or bulk editing.

![](/resources/docs/otter-texteditor.png){height="" width="50%"}

* **Visual Mode:** A graphical user interface for building plans intuitively.

![](/resources/docs/otter-visualeditor.png){height="" width="50%"}

Both modes are fully interchangeable.

## Logging Scope and Readability

All blocks in OtterScript create log scopes:

* Execution logs are grouped and collapsible.
* [`Log-*`](/docs/executionengine/coreconcepts/statements/general#log-statements) statements make it easy to follow plan execution.
* Short descriptions and structure improve traceability.
* Sensitive info can be hidden with permission settings.

## Execution Modifiers with `with`

OtterScript uses `with` statements to modify execution behavior. These are not a special type of block—they simply wrap logic with additional behavior.

### `with async`

Runs the inner statements in parallel:

```bash
for server localhost
{
    with async
    {
        Log-Information "Running asynchronously...";
    }
}
await;
Log-Information "Async blocks completed.";
```

**Gotcha:** Use `await;` to pause until all async statements finish.

### `with lock`

Ensures only one block with the same lock token runs at a time:

```bash
with lock=deploy-lock
{
    Log-Information "This block is locked.";
}
```

Useful when multiple deployments target the same resource.

### `with isolation`

Runs operations in a temporary process for better separation and security:

```bash
with isolation
{
    Log-Information "Running in an isolated process.";
}
```

Available in BuildMaster `6.1.11+` and Otter `2.2.5+`.

### `with credentials`

Runs the statement block as a different user (credential must be defined):

```bash
with credentials=admin-user
{
    Log-Information "Running as admin.";
}
```