---
title: CWE-352 - Cross-Site Request Forgery (CSRF)
---

<div class="author-title">
<div class="employee-image large kim"></div>
<div>

# Understanding and Preventing Cross-Site Request Forgery in Web Applications
By Kim Pento

</div>
</div>
Cross-Site Request Forgery (CSRF) is a web security vulnerability that tricks a user's browser into executing unwanted actions on a web application where they are authenticated. If a user is authenticated to a site, a different site can make a request to the authenticated site, and the user's browser will send along any authentication cookies, potentially leading to unauthorized actions.

## The Mechanism Behind CSRF

CSRF attacks exploit the trust that a web application has in the user's browser. For example, if a user is logged into a banking application and visits a malicious website, that website can send a request to the banking application on behalf of the user without their knowledge, potentially leading to unauthorized financial transactions.

## Common Scenarios of CSRF Attacks

1. **Unintended Actions**: For instance, changing email addresses, passwords, or even transferring funds.
2. **Exploiting Authenticated Sessions**: The attack relies on the user having an active session with the target site.
3. **Deceptive Requests**: These can be disguised as legitimate requests from the user by the malicious site.

## Preventing CSRF in C# Web Applications

Several strategies can be employed in C# web applications, particularly in ASP.NET, to prevent CSRF attacks:

### 1. Anti-CSRF Tokens

The most common way to prevent CSRF is by using anti-CSRF tokens, also known as synchronizer tokens. These tokens are unique to each session and must be included as a hidden field in forms.

#### Implementing Anti-CSRF Tokens in ASP.NET

ASP.NET MVC has built-in support for anti-CSRF tokens through the `ValidateAntiForgeryToken` attribute and the `@Html.AntiForgeryToken()` helper method.

```csharp
[HttpPost]
[ValidateAntiForgeryToken]
public ActionResult UpdateProfile(UserProfile profile)
{
  // Update profile logic
  return View();
}
```

In the corresponding Razor view:

```html
@using (Html.BeginForm())
{
  @Html.AntiForgeryToken()
  <!-- Form fields -->
  <input type="submit" value="Update Profile" />
}
```

### 2. Validating the Origin of Requests

Validating the origin or referer header on requests to ensure they come from the same origin can prevent CSRF attacks.

```csharp
protected void Application_BeginRequest(object sender, EventArgs e)
{
  string requestOrigin = Request.Headers["Origin"];
  if (!string.IsNullOrEmpty(requestOrigin))
  {
    // Validate the origin
  }
}

protected void Application_EndRequest(object sender, EventArgs e)
{
  if (Response.StatusCode == 403)
  {
    // Handle forbidden request
  }
}
```

### 3. Using Custom Headers

Including a custom header in AJAX requests and validating it on the server can also prevent CSRF. Browsers do not allow cross-origin sites to send custom headers without proper CORS settings.

```javascript
$.ajax({
  url: '/api/updateProfile',
  type: 'POST',
  headers: {
    'X-CSRF-Header': 'YourCustomToken'
  },
  // Other AJAX settings
});
```

On the server side:

```csharp
if (Request.Headers["X-CSRF-Header"] != "YourCustomToken")
{
  // Reject the request
}
```

## Conclusion

Cross-Site Request Forgery poses a significant risk to web applications, but with proper safeguards, such as anti-CSRF tokens, origin checks, and custom headers, its risks can be mitigated. Developers must be vigilant and employ these security measures, especially in applications handling sensitive data, to protect both the application and its users.