---
title: CWE-798 - Use of Hard-coded Credentials
---

<div class="author-title">
<div class="employee-image large tod"></div>
<div>

# Addressing the Risks of Hard-Coded Credentials in C# Applications
By Tod Hoven

</div>
</div>

In software development, embedding credentials directly into source code is a common but risky practice. Hard-coded credentials, such as passwords or API keys, can lead to significant security vulnerabilities if the code is exposed or accessed by unauthorized individuals.

## The Dangers of Hard-Coded Credentials

Embedding credentials in application code can lead to several security risks:

1\. **Source Code Exposure**: If the source code is compromised, so are the credentials.

2\. **Lack of Flexibility**: Changing credentials means redeploying the application.

3\. **Shared Credentials Across Environments**: Using the same credentials in development, testing, and production can lead to serious security breaches.

## Best Practices to Avoid Hard-Coded Credentials in C#

To mitigate these risks, developers should adopt strategies to securely manage and access credentials:

### 1. Using Configuration Files

Store credentials in configuration files outside the source code, and access them securely in your application.

#### Example: Accessing Credentials from Configuration Files

```csharp
// Assume appsettings.json contains the necessary credentials
var configuration = new ConfigurationBuilder()
    .AddJsonFile("appsettings.json")
    .Build();
string databasePassword = configuration["DatabasePassword"];
```

### 2. Environment Variables

Another secure way to handle credentials is by storing them in environment variables.

#### Example: Accessing Environment Variables

```csharp
string apiKey = Environment.GetEnvironmentVariable("API_KEY");
```

### 3. Secure Secrets Management

Use a secrets management tool or service to securely store and access credentials.

#### Example: Using Azure Key Vault

```csharp
var keyVaultClient = new KeyVaultClient(new KeyVaultClient.AuthenticationCallback(secretsManager.GetAccessToken));
var secret = await keyVaultClient.GetSecretAsync("https://myvault.vault.azure.net/secrets/mysecret");
string secretValue = secret.Value;
```

### 4. Using Managed Identity Services

In cloud environments, use managed identity services for secure and automatic credential handling.

#### Example: Azure Managed Identity

```csharp
// Azure Managed Identity automatically handles the credentials
var client = new SecretClient(new Uri("https://myvault.vault.azure.net/"), new DefaultAzureCredential());
KeyVaultSecret secret = client.GetSecret("database-connection-string");
string connectionString = secret.Value;
```

### 5. Regular Credential Rotation

Regularly change credentials and keys, especially when someone who had access leaves the team or project.

### 6. Code Reviews and Audits

Regularly perform code reviews and audits to ensure that no hard-coded credentials are left in the source code.


## Conclusion

Hard-coded credentials are a significant security risk in application development. By utilizing configuration files, environment variables, secrets management tools, managed identity services, and adhering to regular credential rotation and code reviews, developers can effectively mitigate these risks. These practices are crucial for maintaining the security and integrity of applications, especially those handling sensitive data or operations.