---
title: CWE-918 - Server-Side Request Forgery (SSRF)
---

<div class="author-title"><div class="employee-image large pete"></div><div>

# Mitigating Server-Side Request Forgery in C# Web Applications
By Pete Barnum

</div></div>

Server-Side Request Forgery (SSRF) is a security vulnerability that occurs when a web application is tricked into making unexpected requests to internal resources or third-party systems. This can lead to information disclosure, service disruption, and other malicious activities.

## Understanding SSRF

SSRF attacks typically involve an attacker inducing a server to make a request to an unintended destination, possibly bypassing access controls and retrieving or modifying data. The risk is particularly high in applications that interact with URL-based request systems.

## Common SSRF Attack Scenarios

1\. **Internal Network Access**: Accessing or manipulating internal systems that are not directly accessible from the external network.

2\. **Bypassing IP Whitelists**: Using the server's IP address to bypass IP-based authentication systems.

3\. **Data Exfiltration**: Extracting sensitive data from the server or internal network.

## Preventing SSRF in C# Web Applications

Mitigating SSRF involves validating and sanitizing inputs, as well as limiting what the server can request. Here are some strategies:

### 1. Validating URLs

Validate all URLs to ensure they are pointing to allowed destinations before making requests.

#### Example: URL Validation

```csharp
public bool IsValidUrl(string url)
{
  var allowedDomains = new List<string> { "https://trusteddomain.com", "https://api.trusteddomain.com" };
  Uri uri;
  if (Uri.TryCreate(url, UriKind.Absolute, out uri))
  {
    return allowedDomains.Any(domain => uri.Host.EndsWith(domain));
  }
  return false;
}
```

### 2. Restricting IP Addresses

Block requests to internal IP addresses to prevent access to internal network resources.

#### Example: Blocking Internal IPs

```csharp
public bool IsExternalIpAddress(string ipAddress)
{
  IPAddress ip;
  if (IPAddress.TryParse(ipAddress, out ip))
  {
    return !IPAddress.IsLoopback(ip) && !IsLocalIpAddress(ip);
  }
  return false;
}
private bool IsLocalIpAddress(IPAddress ip)
{
  // Define ranges for local IP addresses and check if 'ip' falls in these ranges
}
```

### 3. Using Allowlists for Resources

Implement an allowlist for the resources or APIs that your application can request.

#### Example: Resource Allowlisting

```csharp
var allowedResources = new List<string> { "https://api.trusteddomain.com/data", "https://trusteddomain.com/info" };
public bool IsAllowedResource(string url)
{
  return allowedResources.Contains(url);
}
```

### 4. Limiting Redirects and Forwards

Control and limit the number of redirects or forwards that are allowed to prevent redirection to unintended destinations.

#### Example: Limiting Redirects

```csharp
HttpClientHandler handler = new HttpClientHandler
{
  AllowAutoRedirect = true,
  MaxAutomaticRedirections = 2
};
HttpClient client = new HttpClient(handler);
```

### 5. Server Configuration

Configure the server to restrict outbound requests to untrusted destinations.

#### No Specific Code Example

This involves configuring network firewalls or server settings, rather than a specific code implementation.

## Conclusion

SSRF poses a significant security risk in modern web applications. To mitigate this threat, developers should validate and sanitize all URLs, restrict requests to internal and loopback addresses, use allowlists for external resources, control redirects, and properly configure servers. These practices are essential in protecting C# web applications from SSRF attacks, ensuring the security and integrity of the application and its underlying systems.