---
title: CWE-502 - Deserialization of Untrusted Data
---

<div class="author-title">
<div class="employee-image large kim"></div>
<div>

# Safeguarding Against Deserialization Vulnerabilities in C# Applications
By Kim Pento

</div>
</div>
Deserialization of untrusted data is a significant security concern in software development. It occurs when an application deserializes data from untrusted sources without sufficient validation, potentially leading to serious consequences like remote code execution or data tampering.

## Understanding Deserialization Vulnerabilities

Deserialization is the process of converting data from a byte stream into a usable object. Vulnerabilities arise when an application accepts serialized data from untrusted sources and deserializes it without adequate checks, giving attackers the opportunity to manipulate the serialized data to compromise the application.

## Common Risks Associated with Deserialization

1. **Remote Code Execution**: Malicious objects can execute arbitrary code upon deserialization.
2. **Data Tampering**: Altering serialized objects can manipulate application data.
3. **Denial of Service**: Deserializing large or complex objects can exhaust system resources.

## Strategies for Secure Deserialization in C#

In C# and .NET environments, there are several best practices to mitigate risks associated with deserialization:

### 1. Avoid Deserializing from Untrusted Sources

The primary defense is to not deserialize data from unknown or untrusted sources.

#### Example: Secure Deserialization Practice

```csharp
// Assume 'trustedSource' is a trusted source
byte[] trustedData = GetSerializedDataFromTrustedSource(trustedSource);
var deserializedObject = Deserialize<MyObject>(trustedData);

public T Deserialize<T>(byte[] data)
{
    using (var memoryStream = new MemoryStream(data))
    {
        var binaryFormatter = new BinaryFormatter();
        return (T)binaryFormatter.Deserialize(memoryStream);
    }
}
```

### 2. Implementing Strict Type Checking

During deserialization, implement strict type checking to ensure that the deserialized type is expected and safe.

#### Example: Type Checking During Deserialization

```csharp
public T Deserialize<T>(byte[] data)
{
    using (var memoryStream = new MemoryStream(data))
    {
        var binaryFormatter = new BinaryFormatter();
        binaryFormatter.Binder = new SafeSerializationBinder();
        return (T)binaryFormatter.Deserialize(memoryStream);
    }
}

public class SafeSerializationBinder : SerializationBinder
{
    public override Type BindToType(string assemblyName, string typeName)
    {
        // Validate type here
        // Return type or throw exception if type is unexpected
    }
}
```

### 3. Using Secure Serialization Frameworks

Consider using secure serialization frameworks that are designed to reduce the risk of deserialization vulnerabilities.

#### Example: Using JSON.NET for Safe Deserialization

```csharp
string trustedJson = GetJsonFromTrustedSource(trustedSource);
var deserializedObject = JsonConvert.DeserializeObject<MyObject>(trustedJson);
```

### 4. Input Validation and Sanitization

Validate and sanitize the serialized data before deserialization to ensure it doesn’t contain malicious content.

#### Example: Data Validation Before Deserialization

```csharp
byte[] data = GetSerializedData();
if (IsValidData(data))
{
    var deserializedObject = Deserialize<MyObject>(data);
}

public bool IsValidData(byte[] data)
{
    // Implement validation logic
    // Return true if data is valid, false otherwise
}
```

### 5. Minimizing the Use of Serialization

Minimize the overall use of serialization, especially for passing data within or between applications.

#### Example: Reducing Dependency on Serialization


Instead of serializing objects, consider using data transfer objects (DTOs) or other methods


## Conclusion

Deserialization of untrusted data presents significant risks in C# applications. By avoiding deserialization from untrusted sources, implementing strict type checking, using secure serialization frameworks, performing input validation, and minimizing the use of serialization, developers can greatly mitigate these risks. These practices are essential for developing secure, robust, and reliable software.