---
title: CWE-22 - Improper Limitation of a Pathname to a Restricted Directory ('Path Traversal')
---

<div class="author-title">
<div class="employee-image large kim"></div>
<div>

# Understanding Path Traversal Vulnerability in Web Applications
By Kim Pento

</div>
</div>
Path traversal, often known as directory traversal, is a critical security flaw in web applications. It allows attackers to access files and directories that are stored outside the web root folder. By manipulating variables that reference files with dot-dot-slash (../) sequences and similar methods, an attacker can traverse the file system to access files or directories that are not intended to be accessible.

## How Path Traversal Works

In a typical path traversal attack, the goal is to exploit the security weakness in the way an application constructs file paths. The attacker manipulates a path used in a file operation, aiming to escape out of the directory root set by the application and access files or directories elsewhere on the system.

For example, in a web application where users are supposed to access files within a specific directory, the attacker might use `../` sequences to move up into parent directories and potentially access sensitive files like system configuration or password files.

## Example of Vulnerability in C#

Consider a C# web application that allows users to download files. The application might take a file name as a parameter and concatenate it with a base directory path to form the full file path.

```csharp
public FileStream DownloadFile(string fileName)
{
    string basePath = "/user/files/";
    string fullPath = basePath + fileName;
    return File.OpenRead(fullPath);
}
```

In this scenario, an attacker could manipulate the `fileName` parameter. For instance, using a file name like `../web.config` could potentially give them access to the web server’s configuration file.

## Mitigating Path Traversal

To mitigate this vulnerability, developers need to implement strict input validation and avoid using user-controlled input to access file system paths. Here are some strategies to adopt in C#:

1. **Use Path Functions**: Utilize built-in path manipulation functions that are designed to handle file paths securely.

    ```csharp
    string fileName = "../web.config";
    string safeFileName = Path.GetFileName(fileName); // Returns "web.config"
    ```

2. **Validate Input**: Ensure that the input conforms to expected patterns, and reject any suspicious input.

    ```csharp
    if (!Regex.IsMatch(fileName, @"^[a-zA-Z0-9_\-\.]+$"))
    {
        throw new InvalidOperationException("Invalid file name");
    }
    ```

3. **Use Absolute Path Checks**: Compare the absolute path of the resolved file with the intended base directory to ensure it hasn’t traversed out.

    ```csharp
    string fullPath = Path.GetFullPath(basePath + fileName);
    if (!fullPath.StartsWith(basePath))
    {
        throw new SecurityException("Attempted Path Traversal");
    }
    ```

4. **Implement Access Control**: Ensure that the application enforces proper access control, limiting users to files they are permitted to access.

5. **Regular Audits and Code Reviews**: Regularly audit your codebase for path traversal vulnerabilities and conduct thorough code reviews to catch potential issues.

## Conclusion

Path traversal vulnerabilities pose a significant threat to web application security. It’s essential for developers to understand how these attacks are carried out and take proactive measures to prevent them. By validating input, using path functions appropriately, and regularly reviewing code for potential vulnerabilities, the risk of a path traversal attack can be significantly reduced.